#include <argp.h>
#include <stdbool.h>
#include <string.h>
#include <sys/sysinfo.h>
#include <stdlib.h>

#include "util/kmodules_parser.h"
#include "util/generic_conf_parser.h"
#include "util/sync_control_parser.h"
#include "control/job_scheduler.h"

#ifndef MANUAL_RELEASE_TAG
#include "manual_release_tag.h"
#endif

const char *kmodules_loader_doc =
		"Modules Loader - Loads a set of kernel modules given as command line option or defined in a configuration file.";

/* A description of the arguments we accept. */
const char *kmodules_loader_args_doc = "[<modules list>]";

#ifdef COMP_GIT_VERSION
	#define VERSION_INFO_VERSION "2.0_" COMP_GIT_VERSION "(last update: " MANUAL_RELEASE_TAG ")"
#else
	#define VERSION_INFO_VERSION "2.0_" MANUAL_RELEASE_TAG
#endif

const char *kmodule_loader_program_version = "kmodule_loader, " VERSION_INFO_VERSION;

/* The options we understand. */
static struct argp_option kmodules_loader_options[] = {
		{"max-threads", 'n', "<#threads>" , 0, "Don't use more than <#threads> threads to load the modules.", 0 },
		{"config-file", 'c', "<conf file>", 0, "Load modules defined in given config file. The file can "
				"additionally contain a list of devices kmodule_loader is waiting for to appear after the modules "
				"have been loaded. Finally, access rights of expected device nodes and attributes can be defined. "
				"They are set by kmodule_loader respectively.", 0 },
		{"timeout",     't', "<timeout>",   0, "The timeout in ms until which the kmodule_loader waits for "
				"appearing devices and attributes. This value is set to 1000 by default.", 0 },
		{"kcmdline",    'k', NULL,          0, "Parse modules parameters from kernel commandline.", 0 },
		{"version",	'V', NULL,          0, "Print the version of kmodule_loader.", 0 },
		{"debug",       'd', NULL,          0, "Start kmodule_loader in debug mode.", 0 },
		{ 0 }
};

//-------------------------------------------- properties ------------------------------------------------------------
static unsigned int max_worker_threads_cnt=FALLBACK_WORKER_THREAD_CNT;

static unsigned int passed_kernel_modules=0;

static int timeout_in_ms=DEFAULT_TIMEOUT;

static bool parse_kcmdline = false;
//--------------------------------------------------------------------------------------------------------------------

//-------------------------------------------- private function declaration ------------------------------------------
static unsigned int kmodules_parser_max_worker_threads_cnt_default(void);

static char *kmodules_parser_filter_help(int key, const char *text, void *input);

static error_t kmodules_parser_parse_opt(int key, char *arg, struct argp_state *state);

static error_code_t kmodules_parser_parse_line_in_config(char *line);
//--------------------------------------------------------------------------------------------------------------------

static kml_conf_parser_t kmodules_parser=
{
	.section_name="[kernel modules]",
	.parse_line=kmodules_parser_parse_line_in_config,
	.can_defere_parsing=false
};

static const kml_conf_parser_t *parser_ary[]=
{
		&kmodules_parser,
		&device_parser,
		&attribute_parser,
		NULL
};


//-------------------------------------------- public functions ------------------------------------------------------
logger_loglevel_t kmodules_parser_parse_loglevel(int argc, char *argv[])
{
	int i;
	for (i=1;i<argc;i++)
	{
		if (strcmp(argv[i],"-d")==0 || strcmp(argv[i],"--debug")==0)
			return LOGGER_LEVEL_DEBUG;
	}

	return LOGGER_LEVEL_ERROR;
}

error_code_t kmodules_parser_parse_args(int argc, char *argv[])
{
	error_code_t result=RESULT_OK;

	struct argp argp = {kmodules_loader_options, kmodules_parser_parse_opt,
			kmodules_loader_args_doc, kmodules_loader_doc, NULL, kmodules_parser_filter_help,NULL};

	//initialize config parser
	generic_conf_parser_init(parser_ary);

	//determine default of numbers of worker threads used to load modules
	max_worker_threads_cnt=kmodules_parser_max_worker_threads_cnt_default();
	if (max_worker_threads_cnt==0)
	{
		logger_log_info("MODULES_LOADER - get_nprocs returned 0 cores. Taking %d as number of worker threads if not set "
				"with command line option.", FALLBACK_WORKER_THREAD_CNT);
		max_worker_threads_cnt=FALLBACK_WORKER_THREAD_CNT;
	}

	//parse command line options
	if (argp_parse(&argp, argc, argv, ARGP_NO_EXIT, 0, &result)==EINVAL)
		result=RESULT_INVALID_ARGS;

	logger_log_debug("MODULES_PARSER -> init done (result: %d).", result);
	return result;
}

error_code_t kmodules_parser_parse_config_early(void)
{
	if (generic_conf_parser_file_read_in())
		return generic_conf_parser_parse_early();
	else
		return RESULT_OK;
}

error_code_t kmodules_parser_parse_config_late(void)
{
	if (generic_conf_parser_file_read_in())
		return generic_conf_parser_parse_deferred();
	else
		return RESULT_OK;
}

error_code_t kmodules_parser_parse_kcmdline(const char *kmodule_name, char **parameters)
{
	error_code_t result = RESULT_OK;
	FILE *kcmdline;

	char *line = NULL;
	size_t len = 0;

	if (parse_kcmdline == true)
	{
		char *kmodule_name_point = NULL;
		size_t size = strlen(kmodule_name) + strlen(".") + sizeof(char);

		logger_log_debug("MODULES_LOADER - Parse kcmdline for module %s", kmodule_name);
		kcmdline = fopen("/proc/cmdline", "r");
		if (!kcmdline)
		{
			logger_log_info("MODULES_LOADER - Cannot open kernel command line. Skip parsing kernel command line.");
			return result; // It is not critical when kcmdline cannot be opened and parsed. Let's just continue.
		}

		// create a string "kmodule_name."
		// kmodule_name is acquired from libkmod hence always with underscores. No need to check hyphens.
		kmodule_name_point = (char*) malloc(size);
		if (kmodule_name_point == NULL)
		{
		        return RESULT_NORESOURCES;
		}
		strncpy(kmodule_name_point, kmodule_name, size - 1);

		kmodule_name_point[size - 1] = '\0';
		strncat(kmodule_name_point, ".", size - strlen(kmodule_name_point) - 1);

		while ((getline(&line, &len, kcmdline)) != -1)
		{
			char *ptr = line;
			char *arg = NULL;

			while ((arg = strsep(&ptr, "\t ")) != NULL)
			{
				if (strncmp(arg, kmodule_name_point, size - 1) == 0)
				{
					if (!strchr(arg, '='))
					{
						continue;
					}
					arg += (size - 1);
					result = generic_conf_parser_add_parameters(parameters, arg);
					if (result != RESULT_OK)
					{
						free(kmodule_name_point);
						free(line);
						fclose(kcmdline);

						return result;
					}
				}
			}
		}

		logger_log_debug("MODULES_PARSER -> parsed from kcmdline: %s", *parameters);

		free(kmodule_name_point);
		free(line);
		fclose(kcmdline);

		return result;
	}
	else
	{
		return result;
	}
}

unsigned int kmodules_parser_get_max_worker_threads_cnt(void)
{
	return max_worker_threads_cnt;
}

int kmodules_parser_get_timeout_ms(void)
{
	return timeout_in_ms;
}

unsigned int kmodules_parser_get_scheduled_modules_cnt(void)
{
	return passed_kernel_modules;
}
//--------------------------------------------------------------------------------------------------------------------


//-------------------------------------------- private functions -----------------------------------------------------
static char *kmodules_parser_filter_help(int key, const char *text, void *input)
{
	error_code_t *result_ptr;
	result_ptr=(error_code_t *)input;

	if ((key & ARGP_KEY_HELP_POST_DOC)!=0)
		*result_ptr=RESULT_HELP_PRINTED;
	return (char *)text;
}

static error_t kmodules_parser_parse_opt (int key, char *arg, struct argp_state *state)
{
	char* parse_int_result_ptr;
	error_code_t *result_ptr;
	result_ptr=(error_code_t *)state->input;
	int tmp_int;

	switch (key)
	{
	case 'n':
		tmp_int=strtol(arg,&parse_int_result_ptr,10);
		if (parse_int_result_ptr==arg || tmp_int<=0)
		{
			logger_log_error("Invalid parameter. Option -n and --max-threads must be followed by a positive number.");
			argp_usage(state);
			*result_ptr=RESULT_INVALID_ARGS;
		}
		else
			max_worker_threads_cnt=(unsigned int)tmp_int;
		break;
	case 'c':
	{
		if (!generic_conf_parser_file_read_in())
		{
			*result_ptr=generic_conf_parser_read_file(arg);
		}
		else
		{
			logger_log_error("Command line option -c is allowed to be used once only.");
			argp_usage(state);
			*result_ptr=RESULT_INVALID_ARGS;
		}
		break;
	}
	case 'd':
		//we handled this attribute already at the beginning of main
		break;
	case 't':
		timeout_in_ms=strtol(arg,&parse_int_result_ptr,10);
		if (parse_int_result_ptr==arg || timeout_in_ms<=0)
		{
			logger_log_error("Invalid parameter. Option -t and --timeout must be followed by a positive number.");
			argp_usage(state);
			*result_ptr=RESULT_INVALID_ARGS;
		}
		break;
	case 'k':
		parse_kcmdline = true;
		break;
	case 'V':
		printf("%s\n",kmodule_loader_program_version);
		(*result_ptr)=RESULT_HELP_PRINTED;
		break;
	case ARGP_KEY_ARG:
		passed_kernel_modules++;
		(*result_ptr)=job_scheduler_schedule_module(arg,NULL);
		break;
	case ARGP_KEY_END:
		break;

	default:
		return ARGP_ERR_UNKNOWN;
	}
	return 0;
}

static unsigned int kmodules_parser_max_worker_threads_cnt_default(void)
{
	int tmp=get_nprocs();
	if (tmp<0)
		return 0;
	else
		return (unsigned int)tmp;
}

static error_code_t kmodules_parser_parse_line_in_config(char *line)
{
	char *mod_name;
	char *mod_params;

	mod_name=line;
	mod_params=strchr(mod_name,' ');

	//if we find a space, replace it by '\0' to terminate the module name. Inc the pointer
	//to get the first char of the parameters which must be there because of earlier execution
	//of trimline. This way, mod_params might start with spaces. It is assumed that this is fine for the module
	if (mod_params!=NULL)
	{
		*mod_params='\0';
		mod_params++;
	}

	passed_kernel_modules++;

	//mod_param is either NULL or points to where the parameters start in the string
	return job_scheduler_schedule_module(mod_name, mod_params);
}
//--------------------------------------------------------------------------------------------------------------------
